package PVS.polyhedra;

import Jama.Matrix;
import PVS.Utils.Fmt;


public class Plane {
  
  public Vector3D v;
  public double d;
  public int index;

  // "randon" direction to choose "canonical" plane in case of d == 0
  static Vector3D direction = new Vector3D(3.1415926, 2.718281828459045, 1.718281828459045);

  /** 
      plane is defined by NORMALIZED normal 
      and distance from origin
  */
  public Plane(Vector3D _v, double _d, int index){
    d = _d;
    v = _v;
    this.index = index;
  }
  
  public Plane(Vector3D _v, double _d){
    this(_v,_d,0);
  }
  
  /** 
      plane passing through 3 point
  */
  public Plane(Vector3D v0, Vector3D v1, Vector3D v2){
    this(v0,v1,v2,0);
  }
  public Plane(Vector3D v0, Vector3D v1, Vector3D v2, int index){

    this.index = index;

    Vector3D normal = v2.sub(v1).cross(v0.sub(v1));
    normal.normalize();
    double dot = normal.dot(v1);
    v = normal;
    d = dot;
    if(d < -TOLERANCE){
      d = -d;
      v.x = -v.x;
      v.y = -v.y;
      v.z = -v.z;
    } else if(d < TOLERANCE){
      d = 0; // special case 
      // test for orientation in some "random" direction 
      if(v.dot(direction) < 0){
	v.x = -v.x;
	v.y = -v.y;
	v.z = -v.z;
      }
    }
  }

  /**
     to put this plane into hashtable
   */
  public int hashCode(){
    int value = 
      (int)(3345.563*v.x) +
      (int)(4345.891*v.y) +
      (int)(7341.678*v.z) + 
      (int)(4134.178*d);
    // System.out.println(value);
    return value;
  }

  static final double TOLERANCE=1.e-10;//10;

  public boolean equals(Object o){

    if(o == this)
      return true;
    Plane p = (Plane)o;

    // System.out.println("plane.equals: " + p + " " + this);

    double dx = v.x - p.v.x;if(dx < 0) dx = -dx;
    double dy = v.y - p.v.y;if(dy < 0) dy = -dy;
    double dz = v.z - p.v.z;if(dz < 0) dz = -dz;
    double dd = d - p.d; if(dd < 0) dd = -dd;
    return 
      (dx < TOLERANCE) && 
      (dy < TOLERANCE) && 
      (dz < TOLERANCE) && 
      (dd < TOLERANCE);
  }
  
  static double TOL = 1.e-10;
  static double chop(double x){
    if(x < TOL && x > -TOL)
      return 0;
    else 
      return x;
  }

  public String toString(){
    
    return "{" + Fmt.fmt(chop(v.x),8,5) + " " + 
		   Fmt.fmt(chop(v.y),8,5) + " " + 
		   Fmt.fmt(chop(v.z),8,5) + " " + 
		   Fmt.fmt(chop(d),8,5)+ "}";
  }

  double distance(double x,double y, double z){

    return x*v.x+y*v.y+z*v.z-d;
    
  }

  /**
     return point of intersection of 3 planes, or null, if there is 
     no intersection
   */
  static Vector3D intersect(Plane p1, Plane p2, Plane p3){
    
    try {
      double[][] vals = {{p1.v.x,p1.v.y,p1.v.z},
			 {p2.v.x,p2.v.y,p2.v.z},
			 {p3.v.x,p3.v.y,p3.v.z}};
      double[][] d = {{p1.d},{p2.d},{p3.d}};
      
      Matrix A = new Matrix(vals);
      Matrix B = new Matrix(d);
      Matrix x = A.solve(B);
      double[][] arr = x.getArray();
      
      return new Vector3D(arr[0][0],arr[1][0],arr[2][0]);
    } catch (Exception e){
      
    }
    return null;
  }  

  static Plane getRandomPlane(){
    return new Plane(new Vector3D(Math.random(),Math.random(),Math.random()).normalize(),
		     Math.random(),0);
  }

  public static void main(String[] arg){
    
    int count = 0;
    double maxerr = 0;
    double maxrad = 0;
    for(int i=0; i < 100000; i++){
      Plane p1 = getRandomPlane();
      Plane p2 = getRandomPlane();
      Plane p3 = getRandomPlane();
      Vector3D res = intersect(p1,p2,p3);
      if(res == null){
	count++;
      } else {
	double d1 = p1.distance(res.x,res.y,res.z);
	double d2 = p2.distance(res.x,res.y,res.z);
	double d3 = p3.distance(res.x,res.y,res.z);
	double er = (Math.abs(d1) + Math.abs(d2) + Math.abs(d3));
	if(er > maxerr){
	  maxerr = er;
	  maxrad = res.length();
	}
      }
    }
    System.out.println("no intersection: " + count);
    System.out.println("maxerr: " + maxerr);
    System.out.println("maxrad: " + maxrad);

    /*
    if(res != null)
      System.out.println(Fmt.fmt(res.x,20,17) + Fmt.fmt(res.y,20,17) + Fmt.fmt(res.z,20,17));
    else 
      System.out.println("no intersection");
    */   
  }
}
