package PVS.g3d;

/**
 *
 * Author: Daeron Meyer
 * Copyright (c) 1995 by The Geometry Center, University of Minnesota
 * Distributed under the terms of the GNU Library General Public License
 * 12-14-95
 * Modified by V.Bulatov
 */

import java.applet.Applet;
import java.awt.*;

import java.io.*;
import java.awt.event.*;
import java.awt.image.*;
//import java.net.URL;

import PVS.Utils.WindowUtils;
import PVS.Utils.BorderPanel;
import PVS.Utils.DestroyableFrame;
import PVS.Utils.GraphicsPS;
import PVS.Utils.EventCallback;
import PVS.Utils.TimeoutCallback;
import PVS.Utils.Timeout;
import PVS.Utils.Fmt;
import PVS.Utils.PVSObserver;

/**
 *  Canvas3D: An canvas to display and interact with OFF object 
 */

public class Canvas3D extends Panel implements Runnable{

  static int preferedWidth = 400, preferedHeight = 400;

  PVSObserver observer;

  Model3D	obj;
  boolean	painted = true;
  String	objname = null,
		message = null;
  double	xfac = 0.4;
  // scaleval = 0.1f;
  // xfac = 0.7f * 250 * scaleval;

  Matrix3D	curMatrix = new Matrix3D(),
		tmat = new Matrix3D();

  int		prevx, prevy;

  boolean drawLines = true;
  boolean drawFaces = true;

  PopupMenu displayTypePopup;
  Choice chDisplayType;
  String displayTypeNames[] = {"Normal 3D", "Anaglyph (Red/Blue)", "Anaglyph (Blue/Red)", "Stereo (Parallel)", "Stereo (Crosseyed)"};
  int displayType = NORMAL;
  //double anaglyphAngleGrad = 1.5;
  double anaglyphAngleGrad = 2;
  static final int NORMAL=0,ANAGLYPH_RC = 1,ANAGLYPH_CR = 2, STEREO_PARALLEL=3, STEREO_CROSSED = 4;

  SCanvas canvas;  
  

  public static PrintStream Out = System.out;  
  
  /**
    constructor 

  */
  public Canvas3D(Model3D model){

    
    obj = model;
    initUI();
    init();
    model.setCanvas(this);
  }

  public Canvas3D(){

    initUI();
    init();

  }

  
  public void setObserver(PVSObserver observer){
    this.observer = observer;
  }
  
  public void setModel(Model3D model){

    if(this.obj != null)
      this.obj.clearCanvas(this);

    obj = model;
    init();
    canvas.repaint();

    obj.setCanvas(this);

  }
  /*
  PopupMenu makeDisplayTypePopup(){

    PopupMenu menu = new PopupMenu();
    PopupMenuListener listener = new PopupMenuListener();
    for(int i=0; i < displayTypeNames.length; i++){
      MenuItem mi = new MenuItem(displayTypeNames[i]);
      mi.addActionListener(listener);
      menu.add(mi);
    }
    return menu;
  }
  */
  void initUI(){
    
    this.setBackground(Color.white);
    this.setLayout(new GridBagLayout());
    
    Panel buttonsPanel = new Panel();
    buttonsPanel.setLayout(new GridBagLayout());
    buttonsPanel.setBackground(Color.lightGray);

    Checkbox cbEdges= new Checkbox("Edges", drawLines);
    cbEdges.addItemListener(new EdgesListener());

    Checkbox cbFaces = new Checkbox("Faces", drawFaces);
    cbFaces.addItemListener(new FacesListener());

    //Checkbox cbAnaglyph = new Checkbox("Anaglyph", drawAnaglyph);
    //cbAnaglyph.addItemListener(new AnaglyphListener());

    Button btnReset = new Button("Reset");
    btnReset.addActionListener(new ResetListener());

    Button btnZoomIn = new Button("+");
    btnZoomIn.addMouseListener(new ZoomInListener());
    //btnZoomIn.addActionListener(new ZoomInListener());
    Button btnZoomOut = new Button("-");
    btnZoomOut.addMouseListener(new ZoomOutListener());

    chDisplayType = new Choice();
    for(int i =0; i< displayTypeNames.length; i++){
      chDisplayType.addItem(displayTypeNames[i]);
    }
    chDisplayType.select(displayType);
    chDisplayType.addItemListener(new DisplayTypeListener());
    
    int c= 0;
    WindowUtils.constrain(buttonsPanel,cbEdges,    c++,0,1,1, gbc.NONE, gbc.WEST,0.,0.);
    WindowUtils.constrain(buttonsPanel,cbFaces,    c++,0,1,1, gbc.NONE, gbc.WEST,0.,0.);
    WindowUtils.constrain(buttonsPanel,chDisplayType, c++,0,1,1, gbc.NONE, gbc.WEST,1.,0.,0,0,0,4);
    WindowUtils.constrain(buttonsPanel,btnReset,   c++,0,1,1, gbc.NONE, gbc.WEST,0.,0.);
    WindowUtils.constrain(buttonsPanel,btnZoomIn,  c++,0,1,1, gbc.NONE, gbc.WEST,0.,0.);
    WindowUtils.constrain(buttonsPanel,btnZoomOut, c++,0,1,1, gbc.NONE, gbc.WEST,0.,0.);

    canvas = new SCanvas();

    WindowUtils.constrain(this,buttonsPanel, 0,0,1,1, gbc.HORIZONTAL, gbc.WEST,1.,0.);
    WindowUtils.constrain(this,canvas, 0,1,1,1, gbc.BOTH, gbc.CENTER,1.,1.);    

    canvas.addMouseListener(new MouseListenerClass());
    canvas.addMouseMotionListener(new MouseMotionListenerClass());
    canvas.addKeyListener(new KeyListenerClass());

    //displayTypePopup = makeDisplayTypePopup();
    //this.add(displayTypePopup);

  }

  public Dimension getMinimumSize(){
    return new Dimension(preferedWidth/2, preferedHeight/2);
  }

  public Dimension getPreferredSize(){
    return new Dimension(preferedWidth, preferedHeight);
  }

  /**
    init 
    
   */
  void init(){
      // find bounding box so we can scale  the object to fit
      // in our window
      obj.findBB();				
      double xw = obj.xmax - obj.xmin;		
      double yw = obj.ymax - obj.ymin;		
      double zw = obj.zmax - obj.zmin;		

      if (yw > xw) xw = yw;
      if (zw > xw) xw = zw;

      double f1 = 250 / xw;
      double f2 = 250 / xw;

      // xfac = 0.7f * (f1 < f2 ? f1 : f2) * scaleval;


  }

  double zoomSpeed = 1.02;

  public void zoomIn(){
    xfac *= zoomSpeed;
    canvas.repaint();
  }

  public void zoomOut(){
    xfac /= zoomSpeed;
    canvas.repaint();
  }


  Graphics gc;

  long prevTime = -1;
  double spinSpeed = 0;      
  Vec3 spinAxis = null;
  double spinSpeedCutoff = 0.001;

  /**
     canvas mouse listener 
   */
  class MouseListenerClass extends MouseAdapter implements EventCallback {
    
    public void mousePressed(MouseEvent e) {
      
      if(observer != null){
	if((e.getModifiers() & e.BUTTON1_MASK) != 0){ // && 
             //(e.getModifiers() & (e.CTRL_MASK|e.SHIFT_MASK|e.ALT_MASK)) != 0){
	  
	  // do face pick_up 
	  if(obj instanceof Stellation3D ){
	    int index = ((Stellation3D)obj).findFaceAtPoint(e.getX(), e.getY());
	    
	    if(observer != null && index != -1){
	      Vec3 center = obj.face[index].center;
	      Vec3 normal = obj.normals[obj.face[index].nindex];
	      observer.update(Canvas3D.this, 
			      new Object[]{new double[]{center.x,center.y,center.z},
					   new Integer(e.getModifiers()),
					   new double[]{normal.x,normal.y,normal.z}
				
			      }
			      );
	    }
	  }
	}
      }
      if((e.getModifiers() & e.BUTTON1_MASK) != 0 &&
	 (e.getModifiers() & (e.CTRL_MASK|e.SHIFT_MASK|e.ALT_MASK)) == 0){
        //System.out.println("mouse pressed");
        prevx = e.getX();
        prevy = e.getY();
        spinAxis = null;
        spinSpeed = 0;
        prevTime = System.currentTimeMillis();
        eventCallback = null; 
        
        gc = getGraphics();
      }
      return;      
    } 
    
    public void mouseClicked (MouseEvent e) {
      /*
      if(e.isControlDown()){
	if(e.isShiftDown()){
	  zoomIn();
	} else {
	  zoomOut();
	}
      }
      canvas.repaint();
      */
    }

    /**
      
     */
    public void mouseReleased(MouseEvent e){

      //if((e.getModifiers() & e.BUTTON3_MASK) != 0){

      // Netscape 4.7 has problems with BUTTON1_MASK. Sometimes it is not set
      //} else { // if((e.getModifiers() & e.BUTTON1_MASK) != 0){

        if(spinAxis != null && spinSpeed  > spinSpeedCutoff){
          
          //System.out.println("spinning!");
	  long curTime = System.currentTimeMillis(); // this is rounded to 1/18 sec 
	  if(curTime - prevTime > 500)
	    return;
          eventCallback = this;	
          canvas.repaint();
          prevTime = curTime;
          startFPS = prevTime;
          countFPS = 0;
        } 
	//}
    }

    double averageDt = 0;

    public void processEventCallback(Object who, Object what){
      
      if(spinAxis == null)
        return;        
      
      long curTime = System.currentTimeMillis(); // this is rounded to 1/18 sec 
      
      long dt = curTime - prevTime;
      //System.out.print(" " + dt);
      averageDt = 0.8*averageDt + 0.2*dt*0.001;
      Matrix3D rotation = new Matrix3D(spinAxis, spinSpeed * (averageDt));
      
      curMatrix.mul(rotation);      
      eventCallback = this; 
      prevTime = curTime; 
      
      canvas.repaint();
      
    }
  }

  class MouseMotionListenerClass extends MouseMotionAdapter{

    /**
       mouseDrag
       
    */
    public void mouseDragged(MouseEvent e) {
      
      //System.out.println("mouse dragged: " + e.getModifiers());
      // Netscape 4.7 has problems with BUTTON1_MASK. Sometimes it is not set
      //if((e.getModifiers() & e.BUTTON1_MASK) != 0)
	{
        long curTime = System.currentTimeMillis(); // this is rounded to 1/18 sec
        
        //System.out.println("mouseDragged: " + e);
        int x = e.getX();
        int y = e.getY();
        
        double dx = (x - prevx);
        double dy = (y - prevy);
        double angle = 3*Math.sqrt(dx*dx + dy*dy)/getSize().width;
        spinAxis = new Vec3(dy,-dx, 0);
        spinAxis.normalize();
        Matrix3D rotation = new Matrix3D(spinAxis,angle);
        curMatrix.mul(rotation);
        
        mouseDragged = true;
        canvas.repaint();
        
        double dt = (curTime-prevTime)*0.001;
        if(dt != 0){
          // make averaging 
          spinSpeed = 0.2*angle/dt + 0.8*spinSpeed * Math.exp(-dt*10);
          //System.out.println("ss: " + spinSpeed);
        }
        
        prevx = x; prevy = y;
        prevTime = curTime;
      }
      return;
      
    }
  }


  Image		bbuffer; // double buffering stuff
  Graphics	bgc;
  int Height = -1, Width = -1;

  /** 
    paint 

    transform and paint the object to the graphics context 
    */  
  public void paintCanvas(Graphics g) {

    switch(displayType){
    case NORMAL: 
      drawNormalImage(g);
      break;
    case ANAGLYPH_CR:
      drawAnaglyph(g,anaglyphAngleGrad);
      break;
    case ANAGLYPH_RC:
      drawAnaglyph(g,-anaglyphAngleGrad);
      break;
    case STEREO_PARALLEL:
      drawStereo(g,anaglyphAngleGrad);
      break;
    case STEREO_CROSSED:
      drawStereo(g,-anaglyphAngleGrad);
      break;
    }
    mouseDragged = false;
    if(eventCallback != null){
      // if user is holding down a mouse button
      // this will repeat repeated action 
      EventCallback ec = eventCallback;
      eventCallback = null;
      ec.processEventCallback(null,null);
    }
  }

  boolean drawFPS = false;
  long startFPS = 0;
  long countFPS = 0;
  void drawFPS(Graphics g){
    countFPS ++;
    long time = System.currentTimeMillis() - startFPS;
    if(time == 0)
      return;
    double fps = 0.1*(int)(10*countFPS / (0.001*time));
    g.drawString(Fmt.fmt(fps,6,3), 0, Height - 30);
  }

  void drawNormalImage(Graphics g){

    if(bbuffer == null || getSize().width != Width || getSize().height != Height ){
      Width = getSize().width; Height = getSize().height;
      bbuffer = createImage(Width, Height);
      bgc = bbuffer.getGraphics();			// create image to do
							// double buffering
    }

    if (obj != null) {

	paint(bgc, getSize().width, getSize().height);
	if(drawFPS)
	  drawFPS(bgc);
	
	g.drawImage(bbuffer, 0, 0, this);
	
    } else if (message != null) {
      g.drawString("no data", 3, 20);
    }

  }


  double oldStereoWidth = -1, oldStereoHeight = -1;

  void drawStereo(Graphics g, double angle){

    Dimension dim = canvas.getSize();

    if(oldStereoWidth != dim.width || oldStereoHeight != dim.height){
      oldStereoWidth = dim.width;
      oldStereoHeight = dim.height;
      imgLeft = createImage(dim.width/2, dim.height);
      imgRight = createImage(dim.width/2, dim.height); 
      bbuffer  = createImage(dim.width, dim.height); 
      graphLeft = imgLeft.getGraphics();
      graphRight = imgRight.getGraphics();
      bgc = bbuffer.getGraphics();
    }

    
    drawAnaglyph(graphRight, dim.width/2, dim.height, angle);
    drawAnaglyph(graphLeft, dim.width/2, dim.height, -angle);

    bgc.drawImage(imgLeft, 0, 0, this);
    bgc.drawImage(imgRight, dim.width/2, 0, this);
    g.drawImage(bbuffer, 0, 0, this);
    
  }


  Image imgLeft, imgRight; 
  Graphics graphLeft, graphRight;
  int bufferLeft[];
  int bufferRight[];
  MemoryImageSource imgProducer;
  Image combImage;
  //PaintImageObserver paintImageObserver = new PaintImageObserver();

  /**
     void drawAnaglyph(Graphics g)
   */
  void drawAnaglyph(Graphics g, double angle){

    Dimension dim = canvas.getSize();

    if(imgLeft == null || imgRight == null || 
       imgLeft.getWidth(this) != dim.width || imgLeft.getHeight(this) != dim.height){

      imgLeft = createImage(dim.width, dim.height);
      imgRight = createImage(dim.width, dim.height); 
      graphLeft = imgLeft.getGraphics();
      graphRight = imgRight.getGraphics();
      bufferLeft = new int[dim.width *dim.height];
      bufferRight = new int[dim.width *dim.height];
      imgProducer = new MemoryImageSource(dim.width,dim.height,bufferLeft,0,dim.width);
      imgProducer.setAnimated(true);
      imgProducer.setFullBufferUpdates(true);
      combImage = Toolkit.getDefaultToolkit().createImage(imgProducer);
    }

    makeAnaglyph(angle);    

    imgProducer.newPixels();

    g.drawImage(combImage, 0, 0, this);
    
  }

  void makeAnaglyph(double angle){
    
    Dimension dim = getSize();
    try {

      drawAnaglyph(graphRight, dim.width, dim.height, angle);

      PixelGrabber pgR = 
	new PixelGrabber(imgRight, 0, 0, dim.width, dim.height, bufferRight, 0, dim.width);

      pgR.grabPixels();

      drawAnaglyph(graphLeft, dim.width, dim.height, -angle);
      
      PixelGrabber pgL = 
	new PixelGrabber(imgLeft, 0, 0, dim.width, dim.height, bufferLeft, 0, dim.width);
      pgL.grabPixels();
    } catch (Exception e){
      e.printStackTrace(System.out);
    }
    for(int i =0; i < bufferLeft.length; i++){
      
      //int pixelL = bufferLeft[i];
      //int pixelR = bufferRight[i];
      //int redL =   ((pixelL >> 16) & 0xFF);
      //int greenL = ((pixelL >> 8 ) & 0xFF);
      //int blueL =  ((pixelL      ) & 0xFF); 
      //int redR =   ((pixelR >> 16) & 0xFF);
      //int greenR = ((pixelR >> 8 ) & 0xFF);
      //int blueR =  ((pixelR      ) & 0xFF); 
      //int levelL = (4*redL + 3*greenL + blueL)>>3;
      //int levelR = (4*redR + 3*greenR + blueR)>>3;
      //int res = (levelL + levelR)>>1;
      //bufferLeft[i] = 0xFF000000 | ( levelR << 16)  | levelL;
      int blue = (bufferLeft[i]& 0xFF);
      bufferLeft[i] = 0xFF000000 | ( (bufferRight[i]& 0xFF) << 16)  | (blue << 8) | (blue);
      
    }
  }

  EventCallback eventCallback = null;
  boolean mouseDragged = false;

  /**
    paint 
    
    suitable for printing
   */
  public void paint(Graphics bgc, int width, int height){
    
    obj.mat.unit_flipped();
    /*
    obj.mat.translate(-(obj.xmin + obj.xmax) / 2,
		      -(obj.ymin + obj.ymax) / 2,
   		      -(obj.zmin + obj.zmax));
    */
    obj.mat.mul(curMatrix);    

    int mindiameter = width;
    if(height < width)
      mindiameter = height;

    double scale = ( xfac * mindiameter );

    obj.mat.scale(scale, scale, scale);
    obj.mat.translate(width / 2, height / 2, 0);
    obj.transformed = false;
    
    bgc.setColor(getBackground());
    bgc.fillRect(0,0,width,height);
    obj.paint(bgc);
  }

  public void drawAnaglyph(Graphics bgc, int width, int height, double angle){
    
    obj.mat.unit_flipped();
    //obj.mat.unit();
    /*
    obj.mat.translate(-(obj.xmin + obj.xmax) / 2,
		      -(obj.ymin + obj.ymax) / 2,
   		      -(obj.zmin + obj.zmax));
    */
    obj.mat.mul(curMatrix);    
    // push object under the surface
    obj.mat.translate(0,0,Math.abs((obj.zmax-obj.zmin)/2));
    // rotate for left/right eye
    obj.mat.yrot(angle);
    int mindiameter = width;
    if(height < width)
      mindiameter = height;

    double scale = ( xfac * mindiameter);

    obj.mat.scale(scale, scale, scale);
    obj.mat.translate(width / 2, height / 2, 0);
    obj.transformed = false;
    
    bgc.setColor(Color.gray);
    //bgc.setColor(getBackground());
    bgc.fillRect(0,0,width,height);
    obj.paint(bgc);
  }

  /**
    loadObject

   */
  public static Model3D loadObject(String objname) {

    try {
      // System.out.print("reading: "+objname);System.out.flush();
      Model3D obj = new Model3D ( new FileInputStream(objname));
      return obj;
    } catch(Exception e) {
      e.printStackTrace(System.out);
    }
    // System.out.println(" ready");System.out.flush();
    return null;
  }


  /**
    keyUp

   */
  class KeyListenerClass extends KeyAdapter {

    public void keyTyped(KeyEvent e){
      
      switch(e.getKeyChar()){
      case 'P':
      case 'p':
	doPrint();
	break;
      case 'e':
      case 'E':
      case 'l':
      case 'L':
	drawLines = !drawLines;
	repaint();
	break;
      case 'f':
      case 'F':
	drawFaces = !drawFaces;
	repaint();
	break;

      }
      
    return ;
    }
  }

  Thread thread;
  FileDialog fileDialog; 
  String psName = "stellation.ps";
  
  public void run(){

    try {

      //File file = new File("out.ps");
      //FileOutputStream f = new FileOutputStream(file);      
      OutputStream f = null;
      try{
	if(fileDialog == null){
	  fileDialog = new FileDialog(WindowUtils.getFrame(this), psName, FileDialog.SAVE);
	}
	fileDialog.show();
	if(fileDialog.getFile() == null)
	  return;
	psName = fileDialog.getFile();
	String psDir = fileDialog.getDirectory();
	
	String fileName = psDir + psName;
	File file = new File(fileName);
	f = new FileOutputStream(file);
	Out.println("printing 3D model to file " + fileName);
      } catch(Exception e){
	f = System.out;
	System.out.println("---------start of PS");
	Out.println("printing 3D model to java console");
      }
      //PrintStream f = System.out;
      GraphicsPS ps = new GraphicsPS(f, getGraphics());
      // paint our canvas
      this.paint(ps, ps.getWidth(), ps.getHeight());
      ps.flush(); // important - adds showpage to end of file      
      if(f != System.out){
	f.close();
      } else {
	System.out.println("---------end of PS");
      }
    } catch (Exception e){
      e.printStackTrace(System.out);
    }
  }

  void doPrint(){

    //if(thread != null && thread.isAlive()){
    //  thread.stop();
    //}
    thread = new Thread(this);
    thread.setPriority(Thread.MIN_PRIORITY); 
    thread.start();  
  }    


  class SCanvas extends Canvas {

    /**
       update 
       
    */
    public void update(Graphics g) {
      
      //if (bbuffer == null)
	//g.clearRect(0, 0, getSize().width, getSize().height);      
      paint(g);      
    }

    public void paint(Graphics g){
      paintCanvas(g);
    }

    public Dimension getPreferredSize(){
      return new Dimension(300,300);
    }
  }

  public boolean imageUpdate(Image img,
			     int infoflags,
			     int x,
			     int y,
			     int width,
			     int height){
    return true;
  }

  class EdgesListener implements ItemListener {

    public void itemStateChanged(ItemEvent e){

      if(e.getStateChange() == ItemEvent.SELECTED){
	drawLines = true;
      } else {
	drawLines = false; 
      }
      canvas.repaint();
    }
  }

  class FacesListener implements ItemListener {

    public void itemStateChanged(ItemEvent e){

      if(e.getStateChange() == ItemEvent.SELECTED){
	drawFaces = true;
      } else {
	drawFaces = false;	
      }
      canvas.repaint();
    }
  }
  /*
  class AnaglyphListener implements ItemListener {

    public void itemStateChanged(ItemEvent e){

      if(e.getStateChange() == ItemEvent.SELECTED){
	drawAnaglyph = true;
      } else {
	drawAnaglyph = false;	
      }
      canvas.repaint();
    }
  }
  */
  class ZoomOutListener extends MouseAdapter  implements EventCallback, TimeoutCallback {

    boolean mouseDown = false;
    Timeout timeout;

    public void mousePressed(MouseEvent e){
      mouseDown = true;
      zoomOut();
      // make delay before autorepeat 
      timeout = new Timeout(300, this, null);

    }

    public void mouseReleased(MouseEvent e){

      mouseDown = false;
      eventCallback = null;
      timeout.stop();

    }

    public void timeoutCallback(Object userData){

      if(mouseDown){

	eventCallback = this;       
	zoomOut();
      }
    }

    public void processEventCallback(Object who, Object what){
      if(mouseDown){
	eventCallback = this;       
	zoomOut();
      }
    }

  }

  class ZoomInListener extends MouseAdapter implements EventCallback, TimeoutCallback{

    boolean mouseDown = false;
    Timeout timeout;

    public void mousePressed(MouseEvent e){

      mouseDown = true;
      zoomIn();
      // make delay before autorepeat 
      timeout = new Timeout(300, this, null);

    }

    public void mouseReleased(MouseEvent e){

      mouseDown = false;
      eventCallback = null;
      timeout.stop();
    }

    public void processEventCallback(Object who, Object what){
      if(mouseDown){
	eventCallback = this;       
	zoomIn();
      }
    }

    public void timeoutCallback(Object userData){

      if(mouseDown){

	eventCallback = this;       
	zoomIn();
      }
    }


  }

  class DisplayTypeListener implements ItemListener{

    public void  itemStateChanged(ItemEvent e){
      int ind = chDisplayType.getSelectedIndex();
      displayType = ind;
      //Canvas3D.this.validate();
      canvas.repaint();
    }
    /*
    public void actionPerformed(ActionEvent e){

      Dimension dim = btnDisplayType.getSize();
      displayTypePopup.show(btnDisplayType,0,dim.height);
    }
    */

  }

  class ResetListener implements ActionListener{
    public void actionPerformed(ActionEvent e){
      eventCallback = null;
      curMatrix.unit();
      canvas.repaint();
    }    
  }

  
  class PopupMenuListener implements ActionListener{

    public void actionPerformed(ActionEvent e){
      /*
      String str = e.getActionCommand();
      for(int i =0; i < displayTypeNames.length; i++){
	if(str.equals(displayTypeNames[i])){
	  btnDisplayType.setLabel(str);
	  displayType = i;
	  Canvas3D.this.validate();
	  canvas.repaint();
	}
      }
      */
    }
  }


  static private GridBagConstraints gbc = new GridBagConstraints();

  static Model3D getTestModel (){
    return new Model3D();//vert,faces,edges,poly.colors,poly.icolor);
  }
  /*
  static public void main(String [] args){

    Model3D model = getTestModel();
    
    Frame frame3D = new Frame("3D view");
    frame3D.addWindowListener(frameClosingListener);
    frame3D.setBackground(Color.white);
    canvas3D = new Canvas3D(model);
    canvas3D.Out = Out;
    frame3D.add("Center",canvas3D);
    //frame3D.validate();
    //frame3D.pack();
    frame3D.setBounds(screen.width - screen.height/2,
		      0,screen.height/2,screen.height/2);
    frame3D.validate();
    frame3D.show();
    } 
*/ 

  /**
    main 

   */
  public static void main(String[] args){

    Frame frame = new DestroyableFrame("polyhedra viewer");
    Panel panel = new Panel(); panel.setBackground(Color.lightGray);
    
    GridBagLayout layout = new GridBagLayout();
    frame.setLayout(layout);
    panel.setLayout(layout);
    int nrows = (int)Math.sqrt(args.length);
    int ncolumns = (args.length+nrows-1)/nrows;
    int n = 0;

  main_loop:

    for(int i=0;i < nrows;i++){
      for(int j=0;j < ncolumns;j++){
	if(n>= args.length)
	  break main_loop;
	BorderPanel bp = new BorderPanel();bp.setLayout(layout); 
	bp.setBackground(Color.lightGray);
	WindowUtils.constrain(bp,new Canvas3D(loadObject(args[n])),
                          0,0,1,1, GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.);
	WindowUtils.constrain(panel,bp,
                          j,i,1,1, GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.);
	n++;
      }
    }

    preferedWidth = 600/ncolumns;
    preferedHeight = preferedWidth;

    WindowUtils.constrain(frame,panel,0,0,1,1, GridBagConstraints.BOTH,
                          GridBagConstraints.CENTER,1.,1.);    
    frame.pack();
    frame.show();
  }

}

